//
//  AgoraMusicContentCenter.h
//  AgoraMusicContentCenter
//
//  Created by dingyusong on 2022/6/1.
//  Copyright © 2022 Agora. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "AgoraRtcEngineKit.h"

typedef NS_ENUM(NSUInteger, AgoraMusicContentCenterPreloadStatus) {
    /**
     * 0: No error occurs and preload succeeds.
     */
    AgoraMusicContentCenterPreloadStatusOK = 0,

    /**
     * 1: A general error occurs.
     */
    AgoraMusicContentCenterPreloadStatusError = 1,

    /**
     * 2: The media is preloading.
     */
    AgoraMusicContentCenterPreloadStatusPreloading = 2,
};

typedef NS_ENUM(NSUInteger, AgoraMusicContentCenterStatusCode) {
    /**
     * 0: No error occurs and request succeeds.
     */
    AgoraMusicContentCenterStatusCodeOK = 0,

    /**
     * 1: A general error occurs.
     */
    AgoraMusicContentCenterStatusCodeError = 1,
};

NS_ASSUME_NONNULL_BEGIN

__attribute__((visibility("default"))) @interface MusicChartInfo : NSObject
    /**
     * Name of the music chart
     */
@property (nonatomic, copy) NSString *chartName;
    /**
     * Id of the music chart, which is used to get music list
     */
@property (nonatomic, assign) NSInteger identify;
@end

__attribute__((visibility("default"))) @interface AgoraMvProperty : NSObject
    /**
     * The resolution of the mv
     */
@property (nonatomic, copy) NSString *resolution;
    /**
     * The bandwidth of the mv
     */
@property (nonatomic, copy) NSString *bandwidth;
@end

__attribute__((visibility("default"))) @interface AgoraClimaxSegment : NSObject
    /**
     * The start time of climax segment
     */
@property(nonatomic, assign) NSInteger startTimeMs;
    /**
     * The end time of climax segment
     */
@property(nonatomic, assign) NSInteger endTimeMs;
@end


__attribute__((visibility("default"))) @interface AgoraMusic : NSObject
    /**
     * The songCode of music
     */
@property(nonatomic, assign) NSInteger songCode;
    /**
     * The type of music
     * 1, mp3 with instrumental accompaniment and original
     * 2, mp3 only with instrumental accompaniment
     * 3, mp3 only with original
     * 4, mp4 with instrumental accompaniment and original
     * 5, mv only
     * 6, new type mp4 with instrumental accompaniment and original
     * detail at document of music media center
     */
@property(nonatomic, assign) NSInteger type;
    /**
     * The pitch type of music. 
     * 1, xml lyric has pitch
     * 2, lyric has no pitch
     */
@property(nonatomic, assign) NSInteger pitchType;
    /**
     * The name of music
     */
@property(nonatomic, copy) NSString* name;
    /**
     * The singer of music
     */
@property(nonatomic, copy) NSString* singer;
    /**
     * The poster url of music
     */
@property(nonatomic, copy) NSString* poster;

    /**
     * The release time of music
     */
@property(nonatomic, copy) NSString* releaseTime;
    /**
     * The duration (in seconds) of music
     */
@property(nonatomic, assign) NSInteger durationS;
    /**
     * The lyric list of music
     * 0, xml
     * 1, lrc
     */
@property(nonatomic, strong) NSArray<NSNumber *>* lyricList;
    /**
     * The mv property list of music
     */
@property(nonatomic, strong) NSArray<AgoraMvProperty *>* mvPropertyList;
    /**
     * The climax segment list of music
     */
@property(nonatomic, strong) NSArray<AgoraClimaxSegment *>* climaxSegmentList;
@end

__attribute__((visibility("default"))) @interface AgoraMusicCollection : NSObject
@property(nonatomic, assign) NSInteger count;
@property(nonatomic, assign) NSInteger total;
@property(nonatomic, assign) NSInteger page;
@property(nonatomic, assign) NSInteger pageSize;
@property(nonatomic, strong) NSArray<AgoraMusic *>* musicList;
@end

@protocol AgoraMusicContentCenterEventHandler <NSObject>
    /**
     * The music chart result callback; occurs when getMusicCharts method is called.
     * @param requestId The request id same with return from getMusicCharts.
     * @param status The status of the request. See MusicContentCenterStatusCode
     * @param result The result of music chart collection
     */
- (void)onMusicChartsResult:(NSString *)requestId status:(AgoraMusicContentCenterStatusCode)status result:(NSArray<MusicChartInfo*> *)result;
    /**
     * Music collection, occurs when getMusicCollectionByMusicChartId or searchMusic method is called.
     * @param requestId The request id same with return from getMusicCollectionByMusicChartId or searchMusic
     * @param status The status of the request. See MusicContentCenterStatusCode
     * @param result The result of music collection
     */
- (void)onMusicCollectionResult:(NSString *)requestId status:(AgoraMusicContentCenterStatusCode)status result:(AgoraMusicCollection *)result;
    /**
     * Lyric url callback of getLyric, occurs when getLyric is called
     * @param requestId The request id same with return from getLyric
     * @param lyricUrl  The lyric url of this music
     */
- (void)onLyricResult:(NSString*)requestId lyricUrl:(NSString*)lyricUrl;
    /**
     * preload process callback, occurs when preload is called
     *
     * @param songCode Song code
     * @param percent Preload progress (0 ~ 100)
     * @param status Preload status; see PreloadStatusCode.
     * @param msg The extra information
     * @param lyricUrl  The lyric url of this music
     */
- (void)onPreLoadEvent:(NSInteger)songCode percent:(NSInteger)percent status:(AgoraMusicContentCenterPreloadStatus)status msg:(NSString *)msg lyricUrl:(NSString *)lyricUrl;
@end


@class AgoraRtcEngineKit;
__attribute__((visibility("default"))) @interface AgoraMusicContentCenterConfig : NSObject
@property(assign, nonatomic) AgoraRtcEngineKit* _Nullable rtcEngine;
    /**
     * The app ID of the project that has enabled the music content center
     */
@property (nonatomic, copy) NSString *appId;
    /**
     * music content center need rtmToken to connect with server
     */
@property (nonatomic, copy) NSString *rtmToken;
    /**
     * The user ID when using music content center. It can be different from that of the rtc product.
     */
@property (nonatomic, assign) NSInteger mccUid;
@end

@protocol AgoraMusicPlayerProtocol <AgoraRtcMediaPlayerProtocol>
    /**
    * open a media file with specified parameters.
    *
    * @param songCode The identifier of the media file that you want to play.
    * @param startPos The playback position (ms) of the music file.
    * @return
    * - 0: Success.
    * - < 0: Failure.
    */
- (NSInteger)openMediaWithSongCode:(NSInteger)songCode startPos:(NSInteger)startPos;
@end


__attribute__((visibility("default"))) @interface AgoraMusicContentCenter : NSObject

/**
*  Create an AgoraMusicContentCenter instance.
*  @param config    Configurations for the AgoraMusicContentCenter instance. For details, see AgoraMusicContentCenterConfig.
*/
+ (instancetype)sharedContentCenterWithConfig:(AgoraMusicContentCenterConfig *)config;

/**
*  Register event handler, only the last eventHandler is working.
*/
- (NSInteger)registerEventHandler:(id<AgoraMusicContentCenterEventHandler>)eventHandler;

/**
*  Unregister event handler.
*/
- (NSInteger)unregisterEventHandler;

/**
 * Creates a music player source object and return its pointer.
 * @return
 * - The pointer to \ref rtc::IMusicPlayer "IMusicPlayer",
 *   if the method call succeeds.
 * - The empty pointer NULL, if the method call fails.
 */
- (id<AgoraMusicPlayerProtocol>)createMusicPlayerWithDelegate:(id<AgoraRtcMediaPlayerDelegate>_Nullable)delegate;

/**
 * Get hot list type of music.
 * @return
 * - 0: Success.
 * - < 0: Failure.
 */
- (NSString *)getMusicCharts;

/**
 * Get hot music list by hotType and page info..
 * @param musicChartId The music chart id obtained from getMusicCharts.
 * @param page The page of the music chart, starting from 1
 * @param pageSize The page size, max is 50.
 * @param jsonOption The ext param, default is null.
 * @return
 * - 0: Success.
 * - < 0: Failure.
 */
- (NSString *)getMusicCollectionWith:(NSInteger)musicChartId page:(NSInteger)page pageSize:(NSInteger)pageSize jsonOption:(NSString * _Nullable)jsonOption;

/**
 * Get hot music list by hotType and page info.
 * @param keyWord The key word to search.
 * @param page The page of the music search result, starting from 1
 * @param pageSize The page size, max is 50.
 * @param jsonOption The ext param, default is null.
 * @return
 * - 0: Success.
 * - < 0: Failure.
 */
- (NSString *)searchMusicWith:(NSString *)keyWord page:(NSInteger)page pageSize:(NSInteger)pageSize jsonOption:(NSString * _Nullable)jsonOption;

/**
 * preload a media file with specified parameters.
 *
 * @param songCode The identify of the media file that you want to play.
 * @param option The ext param, default is null.
 * @return
 * - 0: Success.
 * - < 0: Failure.
 */
- (NSInteger)preloadWith:(NSInteger)songCode jsonOption:(NSString* _Nullable)jsonOption;

/**
 * preload a media file with specified parameters.
 *
 * @param songCode The identify of the media file that you want to play.
 * @return
 * - 0: Success.
 * - < 0: Failure.
 */
- (NSInteger)isPreloadedWith:(NSInteger)songCode;

/**
 * get lyric of the song.
 *
 * @param songCode The identify of the media file that you want to play.
 * @param LyricType The type of the lyric file. may be 0:xml or 1:lrc.
 * @return
 * - 0: Success.
 * - < 0: Failure.
 */
- (NSString *)getLyricWith:(NSInteger)songCode lyricType:(NSInteger)lyricType ;

- (void)enableMainQueueDispatch:(BOOL)enabled;

+ (void)destroy;

@end

NS_ASSUME_NONNULL_END
